#ifndef NVM_H
#define NVM_H
#include "status_codes.h"
#include <stdbool.h>
#include "sam.h"
#include <string.h>
#ifdef __cplusplus
extern "C" {
#endif

#define WORDS(x)              ((int)((x) / sizeof(uint32_t)))
#define NVMCTRL_ROW_PAGES		16
enum nvm_command {
	/** Erases the addressed memory row */
	NVM_COMMAND_ERASE_ROW                  = NVMCTRL_CTRLB_CMD_EB,

	/** Write the contents of the page buffer to the addressed memory page */
	NVM_COMMAND_WRITE_PAGE                 = NVMCTRL_CTRLB_CMD_WP,

	/** Erases the addressed auxiliary memory row.
	 *
	 *  \note This command can only be given when the security bit is not set.
	 */
	NVM_COMMAND_ERASE_AUX_ROW              = NVMCTRL_CTRLB_CMD_EP,

	/** Write the contents of the page buffer to the addressed auxiliary memory
	 *  row.
	 *
	 *  \note This command can only be given when the security bit is not set.
	 */
	NVM_COMMAND_WRITE_AUX_ROW              = NVMCTRL_CTRLB_CMD_WP,

	/** Locks the addressed memory region, preventing further modifications
	 *  until the region is unlocked or the device is erased
	 */
	NVM_COMMAND_LOCK_REGION                = NVMCTRL_CTRLB_CMD_LR,

	/** Unlocks the addressed memory region, allowing the region contents to be
	 *  modified
	 */
	NVM_COMMAND_UNLOCK_REGION              = NVMCTRL_CTRLB_CMD_UR,

	/** Clears the page buffer of the NVM controller, resetting the contents to
	 *  all zero values
	 */
	NVM_COMMAND_PAGE_BUFFER_CLEAR          = NVMCTRL_CTRLB_CMD_PBC,

	/** Sets the device security bit, disallowing the changing of lock bits and
	 *  auxiliary row data until a chip erase has been performed
	 */
	NVM_COMMAND_SET_SECURITY_BIT           = NVMCTRL_CTRLB_CMD_SSB,

	/** Enter power reduction mode in the NVM controller to reduce the power
	 *  consumption of the system
	 */
	NVM_COMMAND_ENTER_LOW_POWER_MODE       = NVMCTRL_CTRLB_CMD_SPRM,

	/** Exit power reduction mode in the NVM controller to allow other NVM
	 *  commands to be issued
	 */
	NVM_COMMAND_EXIT_LOW_POWER_MODE        = NVMCTRL_CTRLB_CMD_CPRM,
};

enum status_code nvm_write_buffer
(
	const uint32_t destination_address,
	const uint8_t *buffer,
	uint16_t length
);

enum status_code nvm_read_buffer
(
	const uint32_t source_address,
	uint8_t *const buffer,
	uint16_t length
);

enum status_code nvm_update_buffer
(
	const uint32_t destination_address,
	uint8_t *const buffer,
	uint16_t offset,
	uint16_t length
);

enum status_code nvm_erase_row
(
	const uint32_t row_address
);

enum status_code nvm_execute_command
(
	const enum nvm_command command,
	const uint32_t address,
	const uint32_t parameter
);

bool nvm_is_page_locked
(
	uint16_t page_number
);
void nvm_init
(
	void
);
#ifdef __cplusplus
}
#endif
#endif